// background.js
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    if (request.action === 'downloadImages') {
        console.log('Background: ได้รับคำสั่งดาวน์โหลด', request);
        // ส่ง response กลับทันที
        sendResponse({ success: true });
        // เริ่มดาวน์โหลดหลังจากส่ง response แล้ว
        downloadImages(request.images, request.downloadPath, request.askLocation);
    }
    // ไม่ต้อง return true เพราะเราไม่ต้องการ async response
    return false;
});

async function downloadImages(images, customPath, askLocation) {
    console.log('Background: เริ่มดาวน์โหลด', images.length, 'รูปภาพ');
    console.log('Background: Custom path:', customPath);
    console.log('Background: Ask location:', askLocation);

    let downloadPath = customPath || '';
    let shouldAskLocation = askLocation !== undefined ? askLocation : false;

    try {
        // ดึงการตั้งค่าจาก storage (fallback)
        const result = await chrome.storage.sync.get(['downloadPath', 'askLocation']);
        console.log('Background: ดึงการตั้งค่าจาก storage:', result);

        // ใช้ path ที่ส่งมาก่อน ถ้าไม่มีจึงใช้จาก storage หรือ default
        downloadPath = customPath || result.downloadPath || '';
        // ใช้ askLocation ที่ส่งมาก่อน ถ้าไม่มีจึงใช้จาก storage หรือ default เป็น false
        shouldAskLocation = askLocation !== undefined ? askLocation : (result.askLocation || false);
    } catch (error) {
        console.error('Background: ไม่สามารถเข้าถึง storage:', error);
        // ใช้ค่า default ถ้าไม่สามารถเข้าถึง storage ได้
        downloadPath = customPath || '';
        shouldAskLocation = askLocation !== undefined ? askLocation : false;
    }

    // แก้ไข path ที่ขึ้นต้นด้วย / (absolute path) สำหรับ macOS/Linux
    if (downloadPath.startsWith('/') || downloadPath.startsWith('C:\\') || downloadPath.startsWith('D:\\')) {
        // Chrome ไม่รองรับ absolute path ดังนั้นต้องใช้เฉพาะ relative path
        // แปลงจาก /Users/max/Downloads/downloadhub หรือ C:\Users\max\Downloads\downloadhub เป็น downloadhub
        const pathParts = downloadPath.split(/[\/\\]/);
        downloadPath = pathParts[pathParts.length - 1] || '';
        console.log('Background: แปลง absolute path เป็น relative:', downloadPath);
    }

    console.log('Background: ใช้ download path:', downloadPath);
    console.log('Background: ใช้ ask location:', shouldAskLocation);

    let successCount = 0;
    let errorCount = 0;

    for (let i = 0; i < images.length; i++) {
        const image = images[i];

        try {
            console.log(`Background: กำลังดาวน์โหลด ${i + 1}/${images.length}:`, image.filename);

            // ดาวน์โหลดรูปภาพ
            const downloadId = await new Promise((resolve) => {
                try {
                    chrome.downloads.download({
                        url: image.url,
                        filename: shouldAskLocation ? image.filename : `${downloadPath ? downloadPath + '/' : ''}${image.filename}`,
                        conflictAction: 'overwrite',
                        saveAs: shouldAskLocation
                    }, (downloadId) => {
                        if (chrome.runtime.lastError) {
                            console.error('Background: ข้อผิดพลาดในการดาวน์โหลด:', chrome.runtime.lastError);
                            resolve(null);
                        } else {
                            resolve(downloadId);
                        }
                    });
                } catch (error) {
                    console.error('Background: เกิดข้อผิดพลาดในการเรียกใช้ chrome.downloads:', error);
                    resolve(null);
                }
            });

            if (downloadId) {
                successCount++;
                console.log(`Background: ดาวน์โหลดเรียบร้อย (ID: ${downloadId}):`, image.filename);
            } else {
                errorCount++;
                console.error(`Background: ไม่สามารถดาวน์โหลด:`, image.filename);
            }

            // หน่วงเวลาเล็กน้อยเพื่อไม่ให้ดาวน์โหลดพร้อมกันมากเกินไป
            await new Promise(resolve => setTimeout(resolve, 200));

        } catch (error) {
            errorCount++;
            console.error(`Background: เกิดข้อผิดพลาดในการดาวน์โหลด ${image.filename}:`, error);
        }
    }

    console.log(`Background: เสร็จสิ้น - สำเร็จ: ${successCount}, ล้มเหลว: ${errorCount}`);

    // ส่งผลลัพธ์กลับไปยัง content script
    try {
        const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
        const getBookieTab = tabs.find(tab => tab.url.includes('upshopee.getbookie.com'));
        if (getBookieTab) {
            await chrome.tabs.sendMessage(getBookieTab.id, {
                action: 'downloadComplete',
                success: successCount,
                error: errorCount,
                total: images.length
            });
        }
    } catch (error) {
        console.error('Background: ไม่สามารถส่งผลลัพธ์กลับไปยัง content script:', error);
    }
}