// popup.js
document.addEventListener('DOMContentLoaded', function () {
    const downloadBtn = document.getElementById('downloadBtn');
    const settingsBtn = document.getElementById('settingsBtn');
    const statusDiv = document.getElementById('status');
    const extensionToggle = document.getElementById('extensionToggle');
    const statusText = document.getElementById('statusText');
    const connectionStatus = document.getElementById('connectionStatus');

    console.log('🎮 Popup loaded');

    // โหลดสถานะ Extension
    loadExtensionStatus();

    // ตรวจสอบการเชื่อมต่อกับ content script
    checkConnection();

    // Event Listeners
    extensionToggle.addEventListener('change', toggleExtension);
    downloadBtn.addEventListener('click', downloadImages);
    settingsBtn.addEventListener('click', openSettings);

    // โหลดสถานะการเปิด/ปิด Extension
    function loadExtensionStatus() {
        chrome.storage.sync.get(['extensionEnabled'], function (result) {
            const isEnabled = result.extensionEnabled !== false; // default เป็น true
            extensionToggle.checked = isEnabled;
            updateStatusDisplay(isEnabled);
            updateButtonStates(isEnabled);
        });
    }

    // เปิด/ปิด Extension
    function toggleExtension() {
        const isEnabled = extensionToggle.checked;

        chrome.storage.sync.set({ extensionEnabled: isEnabled }, function () {
            if (chrome.runtime.lastError) {
                console.error('❌ ข้อผิดพลาดในการบันทึก:', chrome.runtime.lastError);
                showStatus('❌ เกิดข้อผิดพลาดในการบันทึกสถานะ', 'error');
                // Revert toggle
                extensionToggle.checked = !isEnabled;
                return;
            }

            updateStatusDisplay(isEnabled);
            updateButtonStates(isEnabled);

            // แจ้งเตือน content script
            notifyContentScript(isEnabled);

            showStatus(
                isEnabled ? '✅ เปิดการทำงาน Extension แล้ว' : '⏹️ ปิดการทำงาน Extension แล้ว',
                isEnabled ? 'success' : 'warning'
            );
        });
    }

    // อัปเดตแสดงสถานะ
    function updateStatusDisplay(isEnabled) {
        statusText.textContent = isEnabled ? '🟢 เปิดใช้งาน' : '🔴 ปิดใช้งาน';
        statusText.className = `status-text ${isEnabled ? 'status-enabled' : 'status-disabled'}`;
    }

    // อัปเดตสถานะปุ่ม
    function updateButtonStates(isEnabled) {
        downloadBtn.disabled = !isEnabled;
        downloadBtn.style.opacity = isEnabled ? '1' : '0.5';
    }

    // แจ้งเตือน content script เรื่องสถานะ
    function notifyContentScript(isEnabled) {
        chrome.tabs.query({ active: true, currentWindow: true }, function (tabs) {
            if (tabs[0]) {
                chrome.tabs.sendMessage(tabs[0].id, {
                    action: 'toggleExtension',
                    enabled: isEnabled
                }, function (response) {
                    if (chrome.runtime.lastError) {
                        console.log('📝 Content script ไม่พร้อม (ปกติสำหรับบางหน้า)');
                    }
                });
            }
        });
    }

    // ตรวจสอบการเชื่อมต่อกับ content script
    function checkConnection() {
        chrome.tabs.query({ active: true, currentWindow: true }, function (tabs) {
            if (!tabs[0]) {
                showConnectionStatus('⚠️ ไม่พบแท็บที่ใช้งาน', 'warning');
                return;
            }

            const url = tabs[0].url;

            // ตรวจสอบ URL ที่ไม่รองรับ
            if (url.startsWith('chrome://') || url.startsWith('chrome-extension://') ||
                url.startsWith('edge://') || url.startsWith('about:')) {
                showConnectionStatus('⚠️ Extension ไม่สามารถทำงานในหน้านี้ได้', 'warning');
                downloadBtn.disabled = true;
                return;
            }

            // ทดสอบการเชื่อมต่อ
            chrome.tabs.sendMessage(tabs[0].id, { action: 'ping' }, function (response) {
                if (chrome.runtime.lastError) {
                    showConnectionStatus('🔄 กรุณารีเฟรชหน้าเว็บแล้วลองใหม่', 'warning');
                } else {
                    showConnectionStatus('✅ เชื่อมต่อสำเร็จ', 'success');
                }
            });
        });
    }

    // แสดงสถานะการเชื่อมต่อ
    function showConnectionStatus(message, type) {
        connectionStatus.textContent = message;
        connectionStatus.className = `connection-status ${type}`;
        connectionStatus.style.display = 'block';

        setTimeout(() => {
            connectionStatus.style.display = 'none';
        }, 4000);
    }

    // ปุ่มดาวน์โหลดรูปภาพ
    function downloadImages() {
        if (!extensionToggle.checked) {
            showStatus('⚠️ กรุณาเปิดใช้งาน Extension ก่อน', 'warning');
            return;
        }

        console.log('🖱️ คลิกปุ่มดาวน์โหลด');

        chrome.tabs.query({ active: true, currentWindow: true }, function (tabs) {
            console.log('📤 ส่งข้อความไปยัง content script');

            chrome.tabs.sendMessage(tabs[0].id, { action: 'downloadImages' }, function (response) {
                if (chrome.runtime.lastError) {
                    console.error('❌ ข้อผิดพลาด:', chrome.runtime.lastError);
                    showStatus('❌ ไม่สามารถเชื่อมต่อกับหน้าเว็บได้ กรุณารีเฟรชหน้าเว็บ', 'error');
                } else if (response && response.success) {
                    console.log('✅ ตอบกลับสำเร็จ:', response);
                    showStatus(`✅ กำลังดาวน์โหลด ${response.count} รูปภาพ`, 'success');
                } else {
                    console.log('⚠️ ตอบกลับ:', response);
                    showStatus('❌ ไม่พบรูปภาพที่มี class "productImage"', 'error');
                }
            });
        });
    }

    // ปุ่มเปิดการตั้งค่า
    function openSettings() {
        chrome.runtime.openOptionsPage();
    }

    // แสดงสถานะ
    function showStatus(message, type) {
        statusDiv.textContent = message;
        statusDiv.className = `status ${type}`;
        statusDiv.style.display = 'block';

        setTimeout(() => {
            statusDiv.style.display = 'none';
        }, 4000);
    }
});