// ฟังก์ชันสำหรับดาวน์โหลดรูปภาพ
function downloadProductImages() {
    if (!extensionEnabled) {
        showNotification('⚠️ Extension ถูกปิดใช้งาน', 'warning');
        return {
            success: false,
            reason: 'disabled'
        };
    }

    const images = document.querySelectorAll('img.productImage');

    if (images.length === 0) {
        showNotification('❌ ไม่พบรูปภาพที่มี class "productImage"', 'error');
        console.log('🖼️ : ไม่พบรูปภาพที่มี class "productImage"');
        return {
            success: false,
            count: 0
        };
    }

    showNotification(`🖼️ พบรูปภาพ ${images.length} รูป กำลังดาวน์โหลด...`, 'info');

    // ส่งข้อมูลรูปภาพไปให้ background script ดาวน์โหลด
    const imageUrls = Array.from(images).map((img, index) => {
        console.log(`Image ${index + 1}: ${img.src}`);
        return {
            url: img.src,
            filename: `productImage_${Date.now()}_${index + 1}.jpg`
        };
    });

    // ดึงการตั้งค่า path ก่อนส่ง
    chrome.storage.sync.get(['downloadPath', 'askLocation'], function (result) {
        console.log('🖼️ : ดึงการตั้งค่า downloadPath:', result.downloadPath);
        console.log('🖼️ : ดึงการตั้งค่า askLocation:', result.askLocation);

        chrome.runtime.sendMessage({
            action: 'downloadImages',
            images: imageUrls,
            downloadPath: result.downloadPath,
            askLocation: result.askLocation
        }, function (response) {
            if (chrome.runtime.lastError) {
                showNotification('❌ เกิดข้อผิดพลาด: ' + chrome.runtime.lastError.message, 'error');
            } else {
                showNotification(`✅ ส่งคำสั่งดาวน์โหลด ${imageUrls.length} รูปภาพเรียบร้อย`, 'success');
            }
        });
    });

    console.log(`กำลังดาวน์โหลด ${images.length} รูปภาพ`);
    return {
        success: true,
        count: images.length
    };
}

(function () {
    'use strict';

    console.log('🖼️ : ✅ Content script loaded!');

    let extensionEnabled = true; // สถานะเริ่มต้น

    // รอให้ DOM โหลดเสร็จก่อน
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initExtension);
    } else {
        initExtension();
    }

    function initExtension() {
        console.log('🖼️ : 🚀 เริ่มต้น Extension');

        // โหลดสถานะ Extension
        loadExtensionStatus();

        // ตรวจสอบ URL และทำงานตามเงื่อนไข
        const currentHostname = window.location.hostname;
        const currentPath = window.location.pathname;

        if (currentHostname === 'seller.shopee.co.th' &&
            currentPath === '/portal/product/new' &&
            extensionEnabled) {
            // สร้างระบบแสดงข้อความแจ้งเตือน
            createNotificationSystem();
            showNotification('🎯 ตรวจพบหน้าเพิ่มสินค้าใหม่ - กำลังเตรียม Text Editor', 'info');
            initShopeeTextEditor();
        } else if ((currentHostname === 'upshopee.getbookie.com' || currentHostname === 'localhost') && extensionEnabled) {
            // สร้างระบบแสดงข้อความแจ้งเตือน
            createNotificationSystem();
            showNotification('🎯 ตรวจพบหน้า Upshopee - พร้อมดาวน์โหลดรูปภาพ', 'info');
            // เพิ่ม Event Listener สำหรับการกดปุ่ม S
            document.addEventListener('keydown', handleKeyPress);
        } else {
            // ปิดการทำงานของ extension ในหน้าอื่นๆ
            extensionEnabled = false;
        }
    }

    // โหลดสถานะ Extension
    function loadExtensionStatus() {
        chrome.storage.sync.get(['extensionEnabled'], function (result) {
            extensionEnabled = result.extensionEnabled !== false; // default เป็น true
            console.log('🖼️ : 📊 สถานะ Extension:', extensionEnabled ? 'เปิด' : 'ปิด');
        });
    }

    // จัดการการกดปุ่ม
    function handleKeyPress(e) {
        if (!extensionEnabled) return; // ไม่ทำงานถ้าปิดใช้งาน

        // ตรวจสอบ URL ก่อนทำงาน
        const currentHostname = window.location.hostname;
        if (currentHostname !== 'upshopee.getbookie.com' && currentHostname !== 'localhost') return;

        // กดปุ่ม S (keyCode 83)
        if (e.key === 's' || e.key === 'S') {
            // ไม่ต้องการให้ทำงานถ้ากำลังพิมพ์ในช่อง input หรือ textarea
            if (e.target.tagName !== 'INPUT' &&
                e.target.tagName !== 'TEXTAREA' &&
                !e.target.isContentEditable) {
                e.preventDefault();
                showNotification('🔍 กำลังค้นหารูปภาพ...', 'info');
                downloadProductImages();
            }
        }
    }

    // รับข้อความจาก popup
    chrome.runtime.onMessage.addListener(function (request, sender, sendResponse) {
        console.log('🖼️ : 📨 ได้รับข้อความ:', request);

        if (request.action === 'downloadImages') {
            if (!extensionEnabled) {
                showNotification('⚠️ Extension ถูกปิดใช้งาน', 'warning');
                sendResponse({ success: false, reason: 'disabled' });
                return;
            }

            showNotification('📥 เริ่มการดาวน์โหลดจาก Popup', 'info');
            const result = downloadProductImages();
            sendResponse(result);

        } else if (request.action === 'downloadComplete') {
            // แสดงผลลัพธ์การดาวน์โหลด
            const { success, error, total } = request;
            if (error > 0) {
                showNotification(`⚠️ ดาวน์โหลดเสร็จ: สำเร็จ ${success}/${total} รูป (ล้มเหลว ${error} รูป)`, 'warning', 6000);
            } else {
                showNotification(`🎉 ดาวน์โหลดเสรัจสิ้น: ${success}/${total} รูปภาพ`, 'success', 5000);
            }

        } else if (request.action === 'toggleExtension') {
            // อัปเดตสถานะการเปิด/ปิด
            extensionEnabled = request.enabled;
            console.log('🖼️ : 🔄 อัปเดตสถานะ Extension:', extensionEnabled ? 'เปิด' : 'ปิด');

            if (extensionEnabled) {
                showNotification('✅ Extension เปิดใช้งานแล้ว', 'success', 2000);
            } else {
                showNotification('⏹️ Extension ปิดใช้งานแล้ว', 'warning', 2000);
            }
            sendResponse({ success: true });

        } else if (request.action === 'test' || request.action === 'ping') {
            console.log('🖼️ : 🧪 ทดสอบการเชื่อมต่อ');
            if (extensionEnabled) {
                showNotification('🔗 เชื่อมต่อกับ Extension สำเร็จ', 'success', 2000);
            }
            sendResponse({ success: true, enabled: extensionEnabled });
        }

        return true; // Keep message channel open for async response
    });

    // ฟังก์ชันสร้างระบบแสดงข้อความแจ้งเตือน
    function createNotificationSystem() {
        // ลบ notification container เก่าถ้ามี
        const oldContainer = document.getElementById('chromeExtNotification');
        if (oldContainer) {
            oldContainer.remove();
        }

        const notificationContainer = document.createElement('div');
        notificationContainer.id = 'chromeExtNotification';
        notificationContainer.style.cssText = `
        position: fixed !important;
        top: 20px !important;
        right: 20px !important;
        z-index: 2147483647 !important;
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif !important;
        max-width: 400px !important;
        pointer-events: none !important;
    `;

        // ตรวจสอบว่า body มีอยู่หรือไม่
        if (document.body) {
            document.body.appendChild(notificationContainer);
            console.log('🖼️ : ✅ สร้าง notification container เรียบร้อย');
        } else {
            // หากยังไม่มี body ให้รอสักครู่
            setTimeout(() => {
                createNotificationSystem();
            }, 100);
        }
    }

    // ฟังก์ชันแสดงข้อความแจ้งเตือน
    function showNotification(message, type = 'info', duration = 1000) {
        console.log(`🔔 Notification: ${message} (${type})`);

        const container = document.getElementById('chromeExtNotification');
        if (!container) {
            // console.error('❌ ไม่พบ notification container');
            createNotificationSystem();
            setTimeout(() => showNotification(message, type, duration), 500);
            return;
        }

        const notification = document.createElement('div');
        const colors = {
            info: { bg: '#2196F3', text: 'white' },
            success: { bg: '#4CAF50', text: 'white' },
            error: { bg: '#f44336', text: 'white' },
            warning: { bg: '#ff9800', text: 'white' }
        };

        const color = colors[type] || colors.info;

        notification.style.cssText = `
        position: fixed !important;
        top: 20px !important;
        right: 20px !important;
        z-index: 2147483647 !important;
        max-width: 400px !important;
        pointer-events: none !important;
        background-color: ${color.bg} !important;
        color: ${color.text} !important;
        padding: 12px 16px !important;
        border-radius: 8px !important;
        margin-bottom: 10px !important;
        box-shadow: 0 4px 12px rgba(0,0,0,0.25) !important;
        font-size: 14px !important;
        font-weight: 500 !important;
        transition: transform 0.3s ease !important;
        pointer-events: auto !important;
        opacity: 1 !important;
        z-index: 2147483647 !important;
        border: 2px solid rgba(255,255,255,0.3) !important;
    `;

        notification.textContent = message;
        container.appendChild(notification);

        // Animation slide in
        setTimeout(() => {
            notification.style.transform = 'translateX(0) !important';
        }, 10);

        // Auto remove
        setTimeout(() => {
            notification.style.transform = 'translateX(100%) !important';
            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 100);
        }, duration);

        console.log(`📝 แสดงข้อความ: ${message}`);
    }

    // ฟังก์ชันสำหรับดาวน์โหลดรูปภาพ
    function downloadProductImages() {
        if (!extensionEnabled) {
            showNotification('⚠️ Extension ถูกปิดใช้งาน', 'warning');
            return { success: false, reason: 'disabled' };
        }

        const images = document.querySelectorAll('img.productImage');

        if (images.length === 0) {
            showNotification('❌ ไม่พบรูปภาพที่มี class "productImage"', 'error');
            console.log('🖼️ : ไม่พบรูปภาพที่มี class "productImage"');
            return { success: false, count: 0 };
        }

        showNotification(`🖼️ พบรูปภาพ ${images.length} รูป กำลังดาวน์โหลด...`, 'info');

        // ส่งข้อมูลรูปภาพไปให้ background script ดาวน์โหลด
        const imageUrls = Array.from(images).map((img, index) => {
            console.log(`Image ${index + 1}: ${img.src}`);
            return {
                url: img.src,
                filename: `${img.dataset.name}.jpg`
            };
        });

        // ดึงการตั้งค่า path ก่อนส่ง
        chrome.storage.sync.get(['downloadPath', 'askLocation'], function (result) {
            console.log('🖼️ : ดึงการตั้งค่า downloadPath:', result.downloadPath);
            console.log('🖼️ : ดึงการตั้งค่า askLocation:', result.askLocation);

            chrome.runtime.sendMessage({
                action: 'downloadImages',
                images: imageUrls,
                downloadPath: result.downloadPath,
                askLocation: result.askLocation
            }, function (response) {
                if (chrome.runtime.lastError) {
                    showNotification('❌ เกิดข้อผิดพลาด: ' + chrome.runtime.lastError.message, 'error');
                } else {
                    showNotification(`✅ ส่งคำสั่งดาวน์โหลด ${imageUrls.length} รูปภาพเรียบร้อย`, 'success');
                }
            });
        });

        console.log(`กำลังดาวน์โหลด ${images.length} รูปภาพ`);
        return { success: true, count: images.length };
    }

    // ฟังก์ชันสำหรับแทรกข้อความใน Shopee Text Editor
    function initShopeeTextEditor() {
        if (!extensionEnabled) return; // ไม่ทำงานถ้าปิดใช้งาน

        // รอให้หน้าเว็บโหลดเสร็จ
        setTimeout(() => {
            showNotification('🔍 กำลังค้นหา Shopee Elements...', 'info');

            // เริ่มการตรวจสอบ elements ต่างๆ
            startShopeeMonitoring();

        }, 2000);
    }

    function startShopeeMonitoring() {
        // ตรวจสอบ Text Editor (.ql-editor)
        fillTextEditor();

        // ตรวจสอบ Product Name Field
        checkProductNameField();

        // ตรวจสอบ Stock Input
        checkStockInput();

        // ตั้ง observer เพื่อคอยตรวจสอบการเปลี่ยนแปลงของ DOM
        const observer = new MutationObserver((mutations) => {
            // ตรวจสอบเฉพาะการเปลี่ยนแปลงที่เกี่ยวข้องกับ stock input
            const stockInput = document.querySelector(`[data-product-edit-field-unique-id="stock"] input`);
            if (stockInput && mutations.some(mutation =>
                mutation.target === stockInput ||
                mutation.target.contains(stockInput) ||
                (mutation.type === 'attributes' && mutation.attributeName === 'value')
            )) {
                checkStockInput();
            }
        });

        observer.observe(document.body, {
            childList: true,
            subtree: true,
            attributes: true,
            attributeFilter: ['value', 'data-product-edit-field-unique-id']
        });

        console.log('🖼️ : 🔄 เริ่มตรวจสอบ Shopee Elements');
    }

    function fillTextEditor() {
        if (!extensionEnabled) return; // ไม่ทำงานถ้าปิดใช้งาน

        const editor = document.querySelector('.ql-editor');

        if (editor && !editor.dataset.filled) {
            showNotification('📝 พบ Text Editor กำลังแทรกข้อความ...', 'info');

            // ดึงข้อความจากการตั้งค่า
            chrome.storage.sync.get(['description'], function (result) {
                console.log('🖼️ : ดึงการตั้งค่า description:', result.description);

                if (result.description) {
                    // แปลงข้อความที่มี \n เป็น <p> tags
                    const paragraphs = result.description.split('\n');
                    const htmlContent = paragraphs.map(paragraph => `<p>${paragraph.trim()}</p>`).join('');

                    console.log('🖼️ : HTML Content:', htmlContent);

                    // แทรกข้อความลงใน editor
                    editor.innerHTML = htmlContent;

                    // ทำเครื่องหมายว่าเติมข้อความแล้ว
                    editor.dataset.filled = 'true';

                    showNotification('✅ แทรกข้อความใน Text Editor เรียบร้อยแล้ว', 'success');
                    console.log('🖼️ : แทรกข้อความใน Shopee Text Editor เรียบร้อยแล้ว');

                    // Trigger input event เพื่อให้ระบบของ Shopee รู้ว่ามีการเปลี่ยนแปลง
                    const event = new Event('input', { bubbles: true });
                    editor.dispatchEvent(event);
                } else {
                    showNotification('⚠️ ไม่พบข้อความ Description ในการตั้งค่า', 'warning');
                }
            });
        }
    }

    function checkProductNameField() {
        if (!extensionEnabled) return;

        const nameInput = document.querySelector('input[data-product-edit-field-unique-id="name"]');

        if (nameInput && !nameInput.dataset.descriptionChecked) {
            // ตรวจสอบว่า input ว่างหรือไม่
            const currentValue = nameInput.value.trim();

            console.log('🖼️ : 🏷️ พบ Product Name Field, ค่าปัจจุบัน:', currentValue);

            if (currentValue === '') {
                // ถ้าว่างแสดงว่าเป็นการกรอกสินค้าใหม่
                showNotification('🆕 ตรวจพบการกรอกสินค้าใหม่ - เติม Description อัตโนมัติ', 'info');

                // หา text editor และเติมข้อความ
                const editor = document.querySelector('.ql-editor');
                if (editor && !editor.dataset.filled) {
                    // รีเซ็ตการตรวจสอบเพื่อให้เติมใหม่
                    editor.dataset.filled = 'false';
                    delete editor.dataset.filled;

                    // เติมข้อความใหม่
                    setTimeout(() => {
                        fillTextEditor();
                    }, 500);
                }
            }

            // ทำเครื่องหมายว่าตรวจสอบแล้ว
            nameInput.dataset.descriptionChecked = 'true';
        }
    }

    function checkStockInput() {
        if (!extensionEnabled) return;

        // ใช้ selector ที่ถูกต้องตามที่แนะนำ
        const stockInput = document.querySelector(`[data-product-edit-field-unique-id="stock"] input`);

        if (stockInput) {
            // ตรวจสอบว่าค่าเป็น 0 หรือว่างเปล่า
            const currentValue = parseInt(stockInput.value.trim(), 10);

            if (!currentValue) {
                if (stockInput.dataset.displayStockError !== 'true') {
                    stockInput.dataset.displayStockError = 'true';

                    // เพิ่ม style เพื่อ highlight
                    stockInput.parentElement.style.cssText = `
                        border: 32px solid #ff0000 !important;
                        box-shadow: 0 0 8px rgba(255, 0, 0, 0.5) !important;
                        background-color: rgba(255, 0, 0, 0.1) !important;
                        animation: pulse 2s infinite !important;
                    `;

                }
            } else {
                if (stockInput.dataset.displayStockError === 'true') {
                    stockInput.dataset.displayStockError = 'false';
                    stockInput.parentElement.style.cssText = `
                        border: 32px solid #00ff00 !important;`;
                }
            }
        }
    }
})();